
/* 
 *
 * Guitar Echo/Reverb Code
 * (c) 2014 SILICON CHIP Publications
 * Author: Nicholas Vinen
 *
 * Chip: PIC32MX450F256H-I/PT
 * Pin assignments:
 *
 *   D+/D-      : USB D+/D-
 *
 *   RB0-RB4    : PMP address lines
 *   RB5/VBUSON : USB VBUS on (active high)
 *   RB6        : PGEC / effect select switch
 *   RB7        : PGED / effect select switch
 *   RB8/AN8    : VR1 (Delay1 adjustment)
 *   RB9-RB15   : PMP address lines
 *   RC12/OSC1  : MCLK from WM8731
 *   RC13/RPC13 : serial audio data to WM8731
 *   RC14/RPC14 : AUX1 (momentary pushbutton to change effects)
 *   RC15       : N/C
 *
 *   RD0/RPD0   : MOSI for WM8731 control SPI
 *   RD1/AN24   : VR2 (Delay2/other adjustment)
 *   RD2/RPD2   : serial audio bit clock from WM8731
 *   RD3/RPD3   : serial audio data from WM8731
 *   RD4/PMWR   : SRAM write enable-bar
 *   RD5/PMRD   : SRAM output enable-bar
 *   RD6        : SRAM CS-bar
 *   RD7        : AUX4 (defeat switch/foot pedal)
 *   RD8/RPD8   : N/C
 *   RD9/RPD9   : serial audio sample clock from WM8731
 *
 *   RE2        : microphone sense switch
 *
 *   RF0/RPF0   : CS-bar for WM8731 control SPI
 *   RF1/RPF1   : SCK for WM8731 control SPI
 */

/*
 TODO: Check effect changes are seamless
 TODO: Check effect transitions are seamless
 */

#include <plib.h>
#include <xc.h>
#include "WM8731.h"
#include "Effects.h"
#include "TrigTables.h"
#include "EEPROM.h"

// DEVCFG3
// USERID = No Setting
#pragma config FSRSSEL = PRIORITY_7     // Shadow Register Set Priority Select (SRS Priority 7)
#pragma config PMDL1WAY = ON            // Peripheral Module Disable Configuration (Allow only one reconfiguration)
#pragma config IOL1WAY = ON             // Peripheral Pin Select Configuration (Allow only one reconfiguration)
#pragma config FUSBIDIO = ON            // USB USID Selection (Controlled by the USB Module)
#pragma config FVBUSONIO = ON           // USB VBUS ON Selection (Controlled by USB Module)

// DEVCFG2
#pragma config FPLLIDIV = DIV_3//DIV_2  // PLL Input Divider (2x Divider for FRC, 3x Divider for CLKI)
#pragma config FPLLMUL = MUL_20         // PLL Multiplier (20x Multiplier)
#pragma config UPLLIDIV = DIV_12        // USB PLL Input Divider (12x Divider)
#pragma config UPLLEN = OFF             // USB PLL Enable (Disabled and Bypassed)
#pragma config FPLLODIV = DIV_1         // System PLL Output Clock Divider (PLL Divide by 1)

// DEVCFG1
#pragma config FNOSC = PRIPLL//FRCPLL   // Oscillator Selection Bits (External clock with PLL or Fast RC Osc with PLL)
#pragma config FSOSCEN = OFF            // Secondary Oscillator Enable (Disabled)
#pragma config IESO = ON                // Internal/External Switch Over (Enabled)
#pragma config POSCMOD = EC             // Primary Oscillator Configuration (External clock/Primary osc disabled)
#pragma config OSCIOFNC = OFF           // CLKO Output Signal Active on the OSCO Pin (Disabled)
#pragma config FPBDIV = DIV_1           // Peripheral Clock Divisor (Pb_Clk is Sys_Clk/1)
#pragma config FCKSM = CSDCMD           // Clock Switching and Monitor Selection (Clock Switch Disable, FSCM Disabled)
#pragma config WDTPS = PS1048576        // Watchdog Timer Postscaler (1:1048576)
#pragma config WINDIS = OFF             // Watchdog Timer Window Enable (Watchdog Timer is in Non-Window Mode)
#pragma config FWDTEN = OFF             // Watchdog Timer Enable (WDT Disabled (SWDTEN Bit Controls))
#pragma config FWDTWINSZ = WISZ_25      // Watchdog Timer Window Size (Window Size is 25%)

// DEVCFG0
#pragma config DEBUG = ON               // Background Debugger Enable (Debugger is Enabled)
#pragma config JTAGEN = ON              // JTAG Enable (JTAG Port Enabled)
#pragma config ICESEL = ICS_PGx2        // ICE/ICD Comm Channel Select (Communicate on PGEC2/PGED2)
#pragma config PWP = OFF                // Program Flash Write Protect (Disable)
#pragma config BWP = OFF                // Boot Flash Write Protect bit (Protection Disabled)
#pragma config CP = OFF                 // Code Protect (Protection Disabled)


#define SYS_FREQ (80000000L)
#define BUFFER_SIZE_WORDS    (65536-1024-1024)

signed short DMA_Buffer[BUFFER_SIZE_WORDS];

void ADCSetup(void) {
    AD1CHSbits.CH0SA = 8;   // input A = AN8
    AD1CHSbits.CH0SB = 24;  // input B = AN24
//    AD1CHS |= 24<<24;       // compensate for buggy header file
    AD1CON1bits.ASAM = 1;   // automatic sampling start
    AD1CON1bits.SSRC = 7;   // auto conversion
    AD1CON1bits.FORM = 0;   // unsigned 16-bit integer result
//    AD1CON2bits.ALTS = 1;   // alternate between inputs A & B
    AD1CON2bits.SMPI = 2;//1;   // interrupt after every three conversions
    AD1CON2bits.CSCNA = 1;  // scan inputs
    AD1CON3bits.ADCS = 255; // maximum TAD = 80MHz / 256 = 3.2us
    AD1CON3bits.SAMC = 31;  // 31 TAD sampling time
    AD1CSSLbits.CSSL5 = 1;
    AD1CSSLbits.CSSL8 = 1;
    AD1CSSLbits.CSSL24 = 1;
    IFS0bits.AD1IF = 0;     // set up interrupt
    IPC5bits.AD1IP = 1;
    IEC0bits.AD1IE = 1;
    AD1CON1bits.ON = 1;     // turn on ADC unit
    AD1CON1bits.SAMP = 1;   // start sampling
}

volatile unsigned short AN5Val, AN8Val, AN24Val;
volatile unsigned long AN5_Accum, AN8_Accum, AN24_Accum, ADCSubsampleCounter, ADCSampleCounter;
void __ISR(_ADC_VECTOR,IPL1) __ADCInterrupt(void) {
    AN5_Accum += ADC1BUF0;
    AN8_Accum += ADC1BUF1;
    AN24_Accum += ADC1BUF2;
    if( ++ADCSubsampleCounter == 8 ) {
        AN5Val = AN5_Accum / 8;
        AN8Val = AN8_Accum / 8;
        AN24Val = AN24_Accum / 8;
        AN5_Accum = 0;
        AN8_Accum = 0;
        AN24_Accum = 0;
        ADCSubsampleCounter = 0;
        ++ADCSampleCounter;
    }
    IFS0bits.AD1IF = 0;
}

unsigned char cur_hpvol;
void UpdateHeadphoneVol() {
    signed short new_hpvol = AN5Val * 80 / 1024 + 1;
    if( new_hpvol != cur_hpvol ) {
        WM8731_Set_Left_Headphone_Out_Vol(new_hpvol - 74, true, true);
        cur_hpvol = new_hpvol;
    }
}

unsigned char Mic_on, Bypass_on;
unsigned long MuteSamples;
static void UpdateMicEnabled() {
    Mic_on = !PORTEbits.RE2;
    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
    MuteSamples = 24000; // mute for half a second after microphone input state changes
}

#define PIP_PERIOD 9600
#define PIP_LEN 3200
unsigned long button_state[4];
unsigned long pip_timer, pip_limit;
unsigned long button_down_delay[3], button_up_delay;
unsigned char button_press, counter, change_index, update_effect_levels;
unsigned short press_AN8val, press_AN24val, switch_timer;
unsigned long save_changes_timer, changes_bitmap, saved_pot_positions[30];
unsigned long EffectLevels[3], EffectGains[3], MinEffectLevel;

static void UpdateEffectLevels() {
    MinEffectLevel = EffectLevels[0];
    if( EffectLevels[1] < MinEffectLevel )
        MinEffectLevel = EffectLevels[1];
    if( EffectLevels[2] < MinEffectLevel )
        MinEffectLevel = EffectLevels[2];

    if( EffectLevels[0] == MinEffectLevel )
        EffectGains[0] = 0x10000;
    else
        EffectGains[0] = (MinEffectLevel << 16) / EffectLevels[0];

    if( EffectLevels[1] == MinEffectLevel )
        EffectGains[1] = 0x10000;
    else
        EffectGains[1] = (MinEffectLevel << 16) / EffectLevels[1];

    if( EffectLevels[2] == MinEffectLevel )
        EffectGains[2] = 0x10000;
    else
        EffectGains[2] = (MinEffectLevel << 16) / EffectLevels[2];
}

static void ReadEEPROMConfig() {
    unsigned int data[32];
    memset(data, 0xFF, sizeof(data));
    if( DataEEReadArray(data, 0, 32) == 0 ) {
        int temp;
        unsigned int saved_data;

        for( temp = 0; temp < 3; ++temp ) {
            EffectType temp_effect;

            for( temp_effect = Echo; temp_effect <= Phaser; ++temp_effect ) {
                g_effects[temp].cur_effect = temp_effect;
                saved_data = data[1 + temp * 10 + temp_effect];
                if( saved_data != 0xFFFFFFFF )
                    Effects_Init(&g_effects[temp], 48000, (saved_data&0xFFF)<<6, ((saved_data>>12)&0xFFF)<<6);
                else
                    Effects_Init(&g_effects[temp], 48000, 0, 0);
            }
            if( data[0] != 0xFFFFFFFF && ((data[0] >> (temp*4))&0xF) <= Phaser )
                g_effects[temp].cur_effect = (data[0] >> (temp*4))&0xF;
            else
                g_effects[temp].cur_effect = (temp == 0 ? Echo : (temp == 1 ? Reverb : Tremolo));
        }

        for( temp = 0; temp < 3; ++temp )
            EffectLevels[temp] = Effects_GetLevel(&g_effects[temp]);
        UpdateEffectLevels();
    }
}

#define BUTTON_DOWN_DELAY 20000
#define BUTTON_UP_DELAY 4000
int which_effect;
int main(void) {
    unsigned char VolumePot_installed;
    unsigned long temp, write_offset;

    DDPCONbits.JTAGEN = 0;                            //  JTAG = OFF
    SYSTEMConfigPerformance(SYS_FREQ);
    INTEnableSystemMultiVectoredInt();
    INTEnableInterrupts();

    DataEEInit();

    ADCSetup();

    // check to see whether we have a pot on AN5/RB5, for volume control
    temp = ADCSampleCounter;
    while( ADCSampleCounter == temp )
        ;
    CNPUBbits.CNPUB5 = 1;
    temp = ADCSampleCounter;
    while( ADCSampleCounter < temp+8+2 )
        ;
    VolumePot_installed = AN5Val < 0x380;
    CNPUBbits.CNPUB5 = 0;
    if( !VolumePot_installed ) {
        CNPDBbits.CNPDB5 = 1;
        temp = ADCSampleCounter;
        while( ADCSampleCounter < temp+8+2 )
            ;
        VolumePot_installed = AN5Val > 0x080;
        CNPDBbits.CNPDB5 = 0;
    }

    // pin 5 of ICSP = PGEC = RB6/AN6 = effect select bit #0
    PORTSetPinsDigitalIn(IOPORT_B, BIT_6);
    ANSELBbits.ANSB6 = 0;
    CNPUBbits.CNPUB6 = 1;
    // pin 4 of ICSP = PGED = RB7/AN7 = effect select bit #1
    PORTSetPinsDigitalIn(IOPORT_B, BIT_7);
    ANSELBbits.ANSB7 = 0;
    CNPUBbits.CNPUB7 = 1;
    // pin 7 of CON5 = RC14 = effect change button
    PORTSetPinsDigitalIn(IOPORT_C, BIT_14);
    CNPDCbits.CNPDC14 = 1;
    // RE2 indicates microphone presence
    PORTSetPinsDigitalIn(IOPORT_E, BIT_2);
    CNPUEbits.CNPUE2 = 1;
    // RD7 = defeat switch
    PORTSetPinsDigitalIn(IOPORT_D, BIT_7);
    CNPUDbits.CNPUD7 = 1;

    temp = ADCSampleCounter;
    while( ADCSampleCounter < temp+8+2 )
        ;

    WM8731_Setup_Control_SPI();
    WM8731_Enable_MCU_Interface(true);
    WM8731_Set_Left_Line_In_Vol(0, false, true);
    if( VolumePot_installed )
        UpdateHeadphoneVol();
    Mic_on = !PORTEbits.RE2;
    Bypass_on = false;
    WM8731_Power_Control(true /* line input */, Mic_on /* mic bias */, true /* adc */, !Bypass_on /* dac */, true /* outputs */, true /* oscillator */, true /* clock output */, false /* power off*/);
    WM8731_Set_Digital_Audio_Interface_Format(Left_Justified_MSB_First, 16, Right_When_LRC_High, false /* DAC left/right swap */, Master_Mode, false /* invert bclk */);
    WM8731_Sampling_Control(256, 48000, 48000, false /* mclk_divide_by_two */, false/*true*/ /* clkout_divide_by_two */);
    WM8731_Set_Analog_Audio_Path(Mic_on /* mic boost */, !Mic_on /* mic mute */, Mic_on ? Microphone : Line_Input /* input */, Bypass_on && !Mic_on /* bypass enable */, !Bypass_on /* dac enable */, Bypass_on && Mic_on /* side tone */, 0 /* side tone attenuation */);
    WM8731_Set_Digital_Audio_Path(false /* adc HPF */, 0 /* de-emphasis */, false /* soft mute */, false /* store HPF DC offset */);

    /* bit clock = SCK1 (RD2), word clock = RPD9, data in = RPD3, data out = RF0 */
    PORTSetPinsDigitalIn(IOPORT_D, BIT_3);
    ANSELDbits.ANSD3 = 0;
    PPSInput(2,SDI1,RPD3);
    PORTSetPinsDigitalIn(IOPORT_D, BIT_9);
    PPSInput(3,SS1,RPD9);
    PORTSetPinsDigitalIn(IOPORT_D, BIT_2);
    ANSELDbits.ANSD2 = 0;
    PORTSetPinsDigitalOut(IOPORT_F, BIT_0);
    PPSOutput(2,RPF0,SDO1);

    SPI1CON = SPI1CON2 = 0;   // Reset settings
    SPI1STATbits.SPIROV = 1; // Clear overflow, if set
    SPI1STATbits.SPITUR = 1; // Clear underflow, if set
    SPI1STAT = 0;
    SPI1BUF;      // Read any data

    SPI1CON2bits.AUDEN = 1;
    SPI1CON2bits.SPISGNEXT = 1;
    SPI1CON2bits.IGNROV = 1;
    SPI1CON2bits.IGNTUR = 1;
    SPI1CON2bits.AUDMOD = 1; // left-justified
    SPI1CONbits.MODE16 = 1; // 16-bit data, 16-bit FIFO, 32-bit channel, 64-bit frame
    SPI1CONbits.SSEN = 1;
    SPI1CONbits.FRMPOL = 1;
    SPI1CONbits.CKP = 1;
    SPI1CONbits.ENHBUF = 1;
    SPI1CONbits.STXISEL = 3;
    SPI1CONbits.SRXISEL = 1;
    SPI1CONbits.ON = 1;

//    PORTSetPinsDigitalIn(IOPORT_D, BIT_7);
//    CNPUDbits.CNPUD7 = 1;

    ReadEEPROMConfig();

    if( !PORTBbits.RB7 ) {
        button_state[0] = 0xFFFFFFFF;
        button_press |= 1;
    }
    if( !PORTBbits.RB6 ) {
        button_state[1] = 0xFFFFFFFF;
        button_press |= 2;
    }
    which_effect = (button_press&3);
    if( which_effect < 2 )
        which_effect ^= 1;
    if( PORTCbits.RC14 ) {
        button_state[2] = 0xFFFFFFFF;
        button_press |= 4;
        button_down_delay[2] = BUTTON_DOWN_DELAY;
    }
    if( !PORTDbits.RD7 ) {
        button_state[3] = 0xFFFFFFFF;
        button_press |= 8;
        which_effect = 0;
    }

    write_offset = 0;
    while(1) {
        if( ++counter == 16 ) {
            counter = 0;
            if( save_changes_timer && --save_changes_timer == 0 ) {
                if( changes_bitmap&0x80000000 ) {
                    if( DataEEWrite(g_effects[0].cur_effect | ((int)g_effects[1].cur_effect<<4) | ((int)g_effects[2].cur_effect<<8), 0) == 0 )
                        changes_bitmap &= ~0x80000000;
                } else if( changes_bitmap & 0x000003FF ) {
                    if( changes_bitmap & (1<<(change_index+10*(1-1))) ) {
                        if( DataEEWrite(saved_pot_positions[10*(1-1) + change_index], 1 + 10*(1-1) + change_index) == 0 )
                            changes_bitmap &= ~(1<<(change_index+10*(1-1)));
                    } else {
                        for( change_index = 0; change_index < 10; ++change_index )
                            if( changes_bitmap & (1<<(change_index+10*(1-1))) )
                                break;
                    }
                } else if( changes_bitmap & 0x000FFC00 ) {
                    if( changes_bitmap & (1<<(change_index+10*(2-1))) ) {
                        if( DataEEWrite(saved_pot_positions[10*(2-1) + change_index], 1 + 10*(2-1) + change_index) == 0 )
                            changes_bitmap &= ~(1<<(change_index+10*(2-1)));
                    } else {
                        for( change_index = 0; change_index < 10; ++change_index )
                            if( changes_bitmap & (1<<(change_index+10*(2-1))) )
                                break;
                    }
                } else if( changes_bitmap&(0x3FF00000) ) {
                    if( changes_bitmap & (1<<(change_index+10*(3-1))) ) {
                        if( DataEEWrite(saved_pot_positions[10*(3-1) + change_index], 1 + 10*(3-1) + change_index) == 0 )
                            changes_bitmap &= ~(1<<(change_index+10*(3-1)));
                    } else {
                        for( change_index = 0; change_index < 10; ++change_index )
                            if( changes_bitmap & (1<<(change_index+10*(3-1))) )
                                break;
                    }
                }
                if( changes_bitmap )
                    save_changes_timer += 1024;
            } else {
                button_state[0] = (button_state[0]<<1) | !PORTBbits.RB7;
                button_state[1] = (button_state[1]<<1) | !PORTBbits.RB6;
                button_state[2] = (button_state[2]<<1) | PORTCbits.RC14;
                button_state[3] = (button_state[3]<<1) | !PORTDbits.RD7;
                if( button_state[0] == 0xFFFFFFFF && !(button_press&1) ) {
                    button_press |= 1;
                    button_down_delay[0] = 0;
                    which_effect = (button_press&3);
                    if( which_effect < 2 )
                        which_effect ^= 1;

                    press_AN8val = AN8Val;
                    press_AN24val = AN24Val;
                } else if( button_state[0] == 0x00000000 && (button_press&1) ) {
                    button_press &= ~1;
                    which_effect = (button_press&3);
                    if( which_effect < 2 )
                        which_effect ^= 1;

                    press_AN8val = AN8Val;
                    press_AN24val = AN24Val;
                } else if( button_press&1 ) {
                    ++button_down_delay[0];
                }
                if( button_state[1] == 0xFFFFFFFF && !(button_press&2) ) {
                    button_press |= 2;
                    button_down_delay[1] = 0;
                    which_effect = (button_press&3);
                    if( which_effect < 2 )
                        which_effect ^= 1;

                    press_AN8val = AN8Val;
                    press_AN24val = AN24Val;
                } else if( button_state[1] == 0x00000000 && (button_press&2) ) {
                    button_press &= ~2;
                    which_effect = (button_press&3);
                    if( which_effect < 2 )
                        which_effect ^= 1;

                    press_AN8val = AN8Val;
                    press_AN24val = AN24Val;
                } else if( button_press&2 ) {
                    ++button_down_delay[1];
                }
                if( button_state[2] == 0xFFFFFFFF && !(button_press&4) ) {
                    button_press |= 4;
                    button_down_delay[2] = 0;
                    button_up_delay = 0;

                    press_AN8val = AN8Val;
                    press_AN24val = AN24Val;
                } else if( button_state[2] == 0x00000000 && (button_press&4) ) {
                    button_press &= ~4;
                    if( button_down_delay[2] < BUTTON_DOWN_DELAY ) {
                        button_up_delay = BUTTON_UP_DELAY;
                        button_down_delay[2] = 0;
                    }
                } else if( update_effect_levels ) {
                    EffectLevels[which_effect-1] = Effects_GetLevel(&g_effects[which_effect-1]);
                    UpdateEffectLevels();
                    update_effect_levels = 0;
                } else if( (button_press&4) && which_effect ) {
                    if( ++button_down_delay[2] >= BUTTON_DOWN_DELAY ||
                            AN8Val >  press_AN8val + 2 ||  AN8Val <  press_AN8val - 2 ||
                            AN24Val > press_AN24val + 2 || AN24Val < press_AN24val - 2 ) {
                        button_down_delay[2] = BUTTON_DOWN_DELAY;
                        if( AN8Val >  press_AN8val + 2 ||  AN8Val <  press_AN8val - 2 ||
                            AN24Val > press_AN24val + 2 || AN24Val < press_AN24val - 2 ) {
                            int pos;

                            Effects_Adjust(&g_effects[which_effect-1], 48000, AN8Val<<6, AN24Val<<6);
                            update_effect_levels = 1;
                            press_AN8val = AN8Val;
                            press_AN24val = AN24Val;

                            save_changes_timer = 12000;
                            pos = (which_effect-1)*10+g_effects[which_effect-1].cur_effect;
                            saved_pot_positions[pos] = (unsigned long)AN8Val | (((unsigned long)AN24Val)<<16);
                            changes_bitmap |= 1<<pos;
                        }
                    }
                }
                if( button_up_delay && --button_up_delay == 0 && which_effect ) {
                    if( ++g_effects[which_effect-1].cur_effect > Phaser )
                        g_effects[which_effect-1].cur_effect = Echo;
                    if( pip_timer == pip_limit ) {
                        pip_timer = 0;
                        pip_limit = PIP_PERIOD * ((int)g_effects[which_effect-1].cur_effect+1) - (PIP_PERIOD - PIP_LEN);
                    }
                    save_changes_timer = 12000;
                    changes_bitmap |= 0x80000000;
                }
                if( button_state[3] == 0xFFFFFFFF && !(button_press&8) ) {
                    button_press |= 8;
                    which_effect = 0;
                } else if( button_state[3] == 0x00000000 && (button_press&8) ) {
                    button_press &= ~8;
                    which_effect = (button_press&3);
                    if( which_effect < 2 )
                        which_effect ^= 1;
                }

                if( VolumePot_installed )
                    UpdateHeadphoneVol();
                if( !PORTEbits.RE2 != Mic_on )
                    UpdateMicEnabled();
            }
        }

        while( 1 ) {
            if( SPI1STATbits.SPIRBE && SPI1STATbits.SPITBE ) {
                SPI1BUF = 0;
                while( SPI1STATbits.SPITBF )
                    ;
                SPI1BUF = 0;
            }
            if( !SPI1STATbits.SPIRBE && !SPI1STATbits.SPITBF ) {
                signed short val, val2;

                val2 = SPI1BUF;
                while( SPI1STATbits.SPIRBE )
                    ;
                val = SPI1BUF;
                if( val < 1024 && val > -1024 && (val2 > 4096 || val2 < -4096) ) {
                    if( ++switch_timer > 1000 ) {
                        while( SPI1STATbits.SPIRBE )
                            ;
                        val = SPI1BUF;
                        while( SPI1STATbits.SPITBF )
                            ;
                        SPI1BUF = 0;
                        switch_timer = 0;
                    }
                } else {
                    if( switch_timer > 0 )
                        --switch_timer;
                }
//                val = ((int)val + (int)val2)>>1;
                DMA_Buffer[write_offset] = val;
                if( MuteSamples ) {
                    val = 0;
                    --MuteSamples;
                } else if( which_effect ) {
                    val = ((unsigned long)Effects_Apply(&g_effects[which_effect-1], DMA_Buffer, write_offset, BUFFER_SIZE_WORDS) * EffectGains[which_effect-1]) >> 16;
                } else {
                    val = ((unsigned long)val * MinEffectLevel) >> 16;
                }
                if( ++write_offset == BUFFER_SIZE_WORDS )
                    write_offset = 0;
                if( pip_timer < pip_limit ) {
                    int phase = pip_timer%PIP_PERIOD;
                    if( phase < PIP_LEN ) {
                        int val2 = val;
                        val2 += SineTable[(phase<<5)&(SINESIZE-1)] / 2;
                        if( val2 > 32767 )
                            val = 32767;
                        else if( val2 < -32768 )
                            val = -32768;
                        else
                            val = val2;
                    }
                    ++pip_timer;
                }

                SPI1BUF = val;
                while( SPI1STATbits.SPITBF )
                    ;
                SPI1BUF = val;
            } else {
                break;
            }
        }
    }
}
